<?php

namespace App\Http\Controllers\api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Laravel\Sanctum\PersonalAccessToken;

class AuthController extends Controller
{
    /**
     * Login API
     */
    public function login(Request $request)
    {
        try {
            $validateUser = Validator::make(
                $request->all(),
                [
                    'email' => 'required|email',
                    'password' => 'required',
                ]
            );

            if ($validateUser->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'validation error',
                    'errors' => $validateUser->errors()
                ], 401);
            }

            if (!Auth::attempt($request->only(['email', 'password']))) {
                return response()->json([
                    'status' => false,
                    'message' => 'Ces informations ne correspondent aucun de nos utilisateurs .',
                ], 401);
            }

            $user = User::where('email', $request->email)->where('role_id', 5)->first();

            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'Ces informations ne correspondent aucun de nos utilisateurs .',
                ], 401);
            }

            $token = $user->createToken("API TOKEN")->plainTextToken;
            
            Log::info('User logged in', [
                'user_id' => $user->id,
                'email' => $user->email,
                'token_id' => explode('|', $token)[0]
            ]);

            return response()->json([
                'status' => true,
                'message' => 'User Logged In Successfully',
                'token' => $token
            ], 200);
        } catch (\Throwable $th) {
            Log::error('Login error: ' . $th->getMessage());
            return response()->json([
                'status' => false,
                'message' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Logout API
     */
    public function logout(Request $request)
    {
        try {
            Log::info('=== LOGOUT ATTEMPT ===');
            Log::info('Headers:', $request->headers->all());
            Log::info('Bearer Token:', ['token' => $request->bearerToken()]);
            
            // Try multiple ways to get the user
            $user1 = $request->user();
            $user2 = Auth::guard('sanctum')->user();
            $user3 = Auth::user();
            
            Log::info('User retrieval attempts:', [
                'request->user()' => $user1 ? $user1->id : null,
                'Auth::guard(sanctum)->user()' => $user2 ? $user2->id : null,
                'Auth::user()' => $user3 ? $user3->id : null,
            ]);

            // Try to find token manually
            $bearerToken = $request->bearerToken();
            if ($bearerToken) {
                $tokenParts = explode('|', $bearerToken);
                if (count($tokenParts) === 2) {
                    $tokenId = $tokenParts[0];
                    $tokenValue = $tokenParts[1];
                    
                    Log::info('Token parts:', [
                        'id' => $tokenId,
                        'value_length' => strlen($tokenValue)
                    ]);
                    
                    // Check if token exists in database
                    $tokenModel = PersonalAccessToken::find($tokenId);
                    if ($tokenModel) {
                        Log::info('Token found in DB:', [
                            'id' => $tokenModel->id,
                            'tokenable_id' => $tokenModel->tokenable_id,
                            'name' => $tokenModel->name,
                            'last_used_at' => $tokenModel->last_used_at
                        ]);
                        
                        // Verify the hash
                        $hashMatches = hash_equals($tokenModel->token, hash('sha256', $tokenValue));
                        Log::info('Hash verification:', ['matches' => $hashMatches]);
                        
                        if ($hashMatches) {
                            $user = User::find($tokenModel->tokenable_id);
                            if ($user) {
                                Log::info('User found via manual token lookup:', ['user_id' => $user->id]);
                                $tokenModel->delete();
                                Log::info('Token deleted successfully');
                                
                                return response()->json([
                                    'status' => true,
                                    'message' => 'User Logged Out Successfully',
                                ], 200);
                            }
                        }
                    } else {
                        Log::error('Token not found in database', ['token_id' => $tokenId]);
                    }
                }
            }

            $user = $request->user();
            
            if (!$user) {
                Log::error('No authenticated user found');
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated',
                ], 401);
            }

            Log::info('Deleting token for user:', ['user_id' => $user->id]);
            $user->currentAccessToken()->delete();
            Log::info('Token deleted successfully');

            return response()->json([
                'status' => true,
                'message' => 'User Logged Out Successfully',
            ], 200);
            
        } catch (\Throwable $th) {
            Log::error('Logout exception:', [
                'message' => $th->getMessage(),
                'file' => $th->getFile(),
                'line' => $th->getLine()
            ]);
            
            return response()->json([
                'status' => false,
                'message' => $th->getMessage()
            ], 500);
        }
    }
}